/**
 * AlsaAudioSinkRecord.cpp
 *
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: D. Girnus / ADITG/SW2 / dgirnus@de.adit-jv.com
*
* \copyright (c) 2014-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/


#include <errno.h>
#include <adit_logging.h>
#include "AlsaAudioSinkRecord.h"

LOG_IMPORT_CONTEXT(aauto_audio)

namespace adit { namespace aauto {


int AlsaAudioSinkRecord::open(std::string fileName, std::string sinkName) {
    int res = 0;
    std::stringstream streamFilename;

    if (fileName.empty() || sinkName.empty()) {
        LOG_ERROR((aauto_audio, "AlsaAudioSinkRecord::open() Input parameter are invalid"));
        res = -1;
    } else {
        /*assign AudioSink name for DLT logging */
        mSinkName.assign(sinkName);

        /* create filename */
        streamFilename << fileName << mFileCount << ".wav";

        /* create and open file */
        mFile = fopen(streamFilename.str().c_str(), "w+");
        if (!mFile) {
            int err = errno;
            LOG_ERROR((aauto_audio, "%s, AlsaAudioSinkRecord() Open file %s failed, errno=%d",
                    mSinkName.c_str(), streamFilename.str().c_str(), err));
            res = -(err);
        } else {
            LOGD_DEBUG((aauto_audio, "%s, AlsaAudioSinkRecord() Open file %s with Id=%p",
                    mSinkName.c_str(), streamFilename.str().c_str(), mFile));

            /* seek to the position behind wav header
               to write the raw sound data */
            fseek(mFile, WAV_HEADER_OFFSET, SEEK_SET);

            /* increase counter for next filename */
            mFileCount++;
        }
    }

    return res;
}

int AlsaAudioSinkRecord::close() {
    int res = 0;

    if (mFile) {
        LOGD_DEBUG((aauto_audio, "%s, AlsaAudioSinkRecord() Close file %p",
                mSinkName.c_str(), mFile));
        fclose(mFile);
        mFile = nullptr;
    }
    return res;
}

int AlsaAudioSinkRecord::write(const void *ptr, unsigned int size) {
    int res = 0;

    if (mFile) {
        res = fwrite(ptr, (unsigned int)size, 1, mFile);
        if (res == 1) {
            LOGD_VERBOSE((aauto_audio, "%s Write data(len=%d) to file %p complete",
                    mSinkName.c_str(), size, mFile));
            res = (int)size;
        } else {
            LOG_WARN((aauto_audio, "%s Write data(len=%d) to file %p failed=%d, errno=%d",
                    mSinkName.c_str(), size, mFile, res, errno));
        }
    } else {
        // TODO: trace mSinkName.c_str() could cause an error if it was not set
        LOG_ERROR((aauto_audio, "%s, AlsaAudioSinkRecord() Write failed because file not open.",
                mSinkName.c_str()));
        res = -1;
    }

    return res;
}

int AlsaAudioSinkRecord::createWavHeader(int codec, int numChannels, int numBitsPerSample, int sampleRate) {
    int res = -1;
    /* add wav header */
    struct  t_wav_header wavHeader;

    if (mFile) {
        int fileSize = getRawDataFileSize();
        if (fileSize > 0) {
            memcpy(&wavHeader.ChunkID[0], "RIFF", 4);
            memcpy(&wavHeader.Format[0], "WAVE", 4);
            memcpy(&wavHeader.Subchunk1ID[0], "fmt ", 4);
            memcpy(&wavHeader.Subchunk2ID[0], "data", 4);

            /* 36 + SubChunk2Size, or more precisely: 4 + (8 + SubChunk1Size) + (8 + SubChunk2Size) */
            wavHeader.ChunkSize = (uint32_t)(36 + fileSize);
            if (MEDIA_CODEC_AUDIO_PCM == codec) {
                /* This is the size of the rest of the Subchunk1 which follows this number */
                wavHeader.Subchunk1Size = (uint32_t)16;    // 16 for PCM

                wavHeader.AudioFormat = (unsigned short)1;      // Audio format 1=PCM
            } else {
                LOG_WARN((aauto_audio, "%s, Different codec than PCM. codec=%d",
                        mSinkName.c_str(), codec));
            }
            wavHeader.Subchunk2Size = (uint32_t)fileSize;
            wavHeader.NumOfChan     = (unsigned short)numChannels;
            wavHeader.bitsPerSample = (unsigned short)numBitsPerSample;
            wavHeader.SamplesPerSec = (uint32_t)sampleRate;
            wavHeader.bytesPerSec   = (uint32_t)(wavHeader.SamplesPerSec * wavHeader.NumOfChan * (wavHeader.bitsPerSample / 8));
            if (1 == wavHeader.NumOfChan) {
                wavHeader.blockAlign = (unsigned short)2;   // 2=16-bit mono
            } else {
                wavHeader.blockAlign = (unsigned short)4;   // 4=16-bit stereo
            }

            /* write wav header at beginning of the recorded wav file */
            res = write(&wavHeader, sizeof(wavHeader));
            if (res == (int)sizeof(wavHeader)) {
                /* write wav header succeeded */
                LOGD_VERBOSE((aauto_audio, "%s,  Write wav header(len=%zu) to file %p succeeded",
                                    mSinkName.c_str(), sizeof(wavHeader), mFile));
                res = 0;
            } else {
                LOG_WARN((aauto_audio, "%s Write wav header(len=%zu) to file %p failed=%d",
                        mSinkName.c_str(), sizeof(wavHeader), mFile, res));
            }
        } else {
            LOG_ERROR((aauto_audio, "%s, AlsaAudioSinkRecord() createWavHeader did not retrieve size=%d of file %p",
                    mSinkName.c_str(), fileSize, mFile));
        }
    } else {
        // TODO: trace mSinkName.c_str() could cause an error if it was not set
        LOG_ERROR((aauto_audio, "%s, AlsaAudioSinkRecord() createWavHeader failed because file not open.",
                mSinkName.c_str()));
    }

    return res;
}

int AlsaAudioSinkRecord::getRawDataFileSize() {
    int fileSize = 0;
    if (mFile) {
        /* seek to end of file */
        fseek(mFile, 0, SEEK_END);
        /* get file size */
        fileSize = ftell(mFile);  // will be Subchunk2Size
        /* substrate the wav header offset
           which was caused at open */
        fileSize -= WAV_HEADER_OFFSET;
        /* seek to beginning of file */
        fseek(mFile, 0, SEEK_SET);
    } else {
        // TODO: trace mSinkName.c_str() could cause an error if it was not set
        LOG_ERROR((aauto_audio, "%s, AlsaAudioSinkRecord() getRawDataFileSize failed because file not open.",
                mSinkName.c_str()));
        fileSize = -1;
    }
    return fileSize;
}

} } /* namespace adit { namespace aauto */


